package presentation;

/**
 * @author Alberto Iachini
 */


import java.awt.GridBagLayout;

import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JLabel;
import java.awt.GridBagConstraints;
import javax.swing.JButton;
import java.awt.Insets;
import org.jgraph.JGraph;

import presentation.graph.GraphFactory;
import presentation.util.Exporter;

import javax.swing.JScrollPane;
import abstraction.Service;
import abstraction.ReadFile;
import java.io.File;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * This class is the panel used to show target service and add and remove this service
 */
public class TargetPanel extends JPanel {

	private static final long serialVersionUID = -4349282040075258598L;
	private OrchestratorFrame owner;
	private JLabel targetLabel = null;
	private JButton addTargetButton = null;
	private JButton deleteTargetButton = null;
	private JButton saveToImageButton1 = null;
	private JButton saveToDotButton1 = null;
	private JLabel statusLabel = null;

	private JScrollPane targetScrollPane = null;
	
	private boolean targetLoaded = false;
	
	private Service targetService;
	
	/**
	 * This method returns the target service
	 * @return the target service
	 */
	public Service getTargetService() {
		return targetService;
	}

	private JGraph targetGraph = null;
	/**
	 * This is the default constructor
	 */
	public TargetPanel(OrchestratorFrame frame) {
		super();
		owner = frame;
		initialize();
	}

	/**
	 * This method initializes this
	 * 
	 */
	private void initialize() {
		GridBagConstraints gridBagConstraints31 = new GridBagConstraints();
		gridBagConstraints31.fill = GridBagConstraints.BOTH;
		gridBagConstraints31.gridy = 1;
		gridBagConstraints31.weightx = 10.0;
		gridBagConstraints31.weighty = 10.0;
		gridBagConstraints31.gridheight = 5;
		gridBagConstraints31.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints31.gridx = 1;
		GridBagConstraints gridBagConstraints5 = new GridBagConstraints();
		gridBagConstraints5.gridx = 1;
		gridBagConstraints5.anchor = GridBagConstraints.WEST;
		gridBagConstraints5.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints5.fill = GridBagConstraints.HORIZONTAL;
		gridBagConstraints5.gridy = 0;
		statusLabel = new JLabel();
		statusLabel.setText("No Target Service Loaded");
		GridBagConstraints gridBagConstraints4 = new GridBagConstraints();
		gridBagConstraints4.gridx = 0;
		gridBagConstraints4.anchor = GridBagConstraints.WEST;
		gridBagConstraints4.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints4.ipadx = 38;
		gridBagConstraints4.gridy = 4;
		GridBagConstraints gridBagConstraints3 = new GridBagConstraints();
		gridBagConstraints3.gridx = 0;
		gridBagConstraints3.anchor = GridBagConstraints.WEST;
		gridBagConstraints3.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints3.ipadx = 49;
		gridBagConstraints3.gridy = 3;
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.gridx = 0;
		gridBagConstraints2.anchor = GridBagConstraints.WEST;
		gridBagConstraints2.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints2.ipadx = 0;
		gridBagConstraints2.gridy = 2;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.gridx = 0;
		gridBagConstraints1.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints1.anchor = GridBagConstraints.WEST;
		gridBagConstraints1.ipadx = 24;
		gridBagConstraints1.gridy = 1;
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.insets = new Insets(10, 10, 10, 10);
		gridBagConstraints.anchor = GridBagConstraints.WEST;
		gridBagConstraints.ipadx = 83;
		gridBagConstraints.gridy = 0;
		targetLabel = new JLabel();
		targetLabel.setText("Target Service");
		this.setSize(300, 200);
		this.setLayout(new GridBagLayout());
		this.add(targetLabel, gridBagConstraints);
		this.add(getAddTargetButton(), gridBagConstraints1);
		this.add(getDeleteTargetButton(), gridBagConstraints2);
		this.add(getSaveToImageButton1(), gridBagConstraints3);
		this.add(getSaveToDotButton1(), gridBagConstraints4);
		this.add(statusLabel, gridBagConstraints5);
		this.add(getTargetScrollPane(), gridBagConstraints31);
	}

	/**
	 * This method initializes addTargetButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getAddTargetButton() {
		if (addTargetButton == null) {
			addTargetButton = new JButton();
			addTargetButton.setText("Add Target Service");
			addTargetButton.setToolTipText("Add a target service");
			addTargetButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					//add a target service
					/*
					 * 1 - get targetFile with JFileChooser
					 * 2 - create targetService with ReadFile
					 * 3 - add Name to status label
					 * 4 - create targetGraph with GraphFactory
					 * 5 - paint the graph
					 * 6 - set the flag targetLoaded to true
					 */
					
					File file = openFile();
					
					if (file!=null) {
						ReadFile.setFileName(file.getAbsolutePath());
						try {
							String target = ReadFile.readFile();
							targetService = new Service(target);
							boolean deterministic = targetService.isDeterministic();
							
							if (deterministic==true) 
							{
								statusLabel.setText("New Target Service Loaded: "+ file.getName());
								targetGraph = (new GraphFactory(targetService)).getGraph();
								targetScrollPane.setViewportView(targetGraph);
								targetLoaded = true;
							}
							else
							{
								targetService= null;
								JOptionPane.showMessageDialog(null, "Target Service must be deterministic", "Warning", JOptionPane.WARNING_MESSAGE);
							}
						} catch (Exception e1) {
							String error = e1.getMessage();
							JOptionPane.showMessageDialog(null, error, "Error",
									JOptionPane.ERROR_MESSAGE);
							//e1.printStackTrace();
						}
					}
				}
			});
		}
		return addTargetButton;
	}

	/**
	 * This method initializes deleteTargetButton	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getDeleteTargetButton() {
		if (deleteTargetButton == null) {
			deleteTargetButton = new JButton();
			deleteTargetButton.setText("Remove Target Service");
			deleteTargetButton.setToolTipText("Remove the target service");
			deleteTargetButton.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					//remove target service
					/*
					 * 1 - set flag targetLoaded to false
					 * 2 - set targetService and targetGraph to null
					 * 3 - delete graph from graphPanel
					 */
					targetLoaded = false;
					statusLabel.setText("No Target Service Loaded");
					targetService = null;
					targetGraph = null;
					targetScrollPane.setViewportView(null);
				}
			});
		}
		return deleteTargetButton;
	}

	/**
	 * This method initializes saveToImageButton1	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToImageButton1() {
		if (saveToImageButton1 == null) {
			saveToImageButton1 = new JButton();
			saveToImageButton1.setText("Save To Image");
			saveToImageButton1.setToolTipText("Save the graph in a jpeg file");
			saveToImageButton1.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					if (targetService != null) {
						//export to image
						File file = saveImageFile();
						if (file != null)
							Exporter.exportToImage(file, targetGraph);
					}
				}
			});
		}
		return saveToImageButton1;
	}

	/**
	 * This method initializes saveToDotButton1	
	 * 	
	 * @return javax.swing.JButton	
	 */
	private JButton getSaveToDotButton1() {
		if (saveToDotButton1 == null) {
			saveToDotButton1 = new JButton();
			saveToDotButton1.setText("Save To DOT File");
			saveToDotButton1.setToolTipText("Save the graph in DOT format");
			saveToDotButton1.addActionListener(new java.awt.event.ActionListener() {
				public void actionPerformed(java.awt.event.ActionEvent e) {
					//export to dot
					if(targetService!=null)
					{
						File file = saveDotFile();
						if(file != null)
							Exporter.exportToDot(file, targetService);
					}
				}
			});
		}
		return saveToDotButton1;
	}


	/**
	 * This method initializes targetScrollPane	
	 * 	
	 * @return javax.swing.JScrollPane	
	 */
	private JScrollPane getTargetScrollPane() {
		if (targetScrollPane == null) {
			targetScrollPane = new JScrollPane();
			targetScrollPane.setViewportView(null);
		}
		return targetScrollPane;
	}
	
	/**
	 * This method shows a JFileChooser to load the target service file
	 * 
	 * @return the file that hold target service
	 */
	private File openFile()
	{
		JFileChooser fc = new JFileChooser();
		
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("Graph File", "txt");
		fc.setFileFilter(filter);
		fc.setMultiSelectionEnabled(false);
		fc.showOpenDialog(owner);
		
		owner.setLastFolder(fc.getCurrentDirectory());

		return fc.getSelectedFile();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into jpeg file
	 * 
	 * @return the selected file
	 */
	private File saveImageFile()
	{
		JFileChooser fc = new JFileChooser();
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("JPEG File", "jpeg");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		
		owner.setLastFolder(fc.getCurrentDirectory());

		return fc.getSelectedFile();
	}
	
	/**
	 * Open a JFileChooser to save the selected graph into dot file
	 * 
	 * @return the selected file
	 */
	private File saveDotFile()
	{
		JFileChooser fc = new JFileChooser();
		
		File dir = owner.getLastFolder();
		if(dir!=null && dir.isDirectory())
			fc.setCurrentDirectory(dir);
		
		FileNameExtensionFilter filter = new FileNameExtensionFilter("DOT File", "dot");
		fc.setFileFilter(filter);
		fc.showSaveDialog(this);
		
		owner.setLastFolder(fc.getCurrentDirectory());

		return fc.getSelectedFile();
	}
	
	/**
	 * this method checks if a target service is been loaded
	 * 
	 * @return true if a target service is been loaded
	 */
	public boolean isTargetLoaded()
	{
		return targetLoaded;
	}
	
	/**
	 * Enable or disable the buttons to load and remove the service
	 * 
	 * @param flag if true enables the buttons
	 */
	public void setEnableButtons(boolean flag)
	{
		this.addTargetButton.setEnabled(flag);
		this.deleteTargetButton.setEnabled(flag);
	}

}
